<?php
/**
 * OptiCore SaaS - Front Controller
 * Punto de entrada único de la aplicación
 */

declare(strict_types=1);

// ── Bootstrap ─────────────────────────────────────────────────
define('DS', DIRECTORY_SEPARATOR);
require_once dirname(__DIR__) . DS . 'config' . DS . 'config.php';

// ── Obtener ruta y método ─────────────────────────────────────
$requestUri    = parse_url($_SERVER['REQUEST_URI'] ?? '/', PHP_URL_PATH);
$basePath      = rtrim(parse_url(BASE_URL, PHP_URL_PATH), '/');
$route         = '/' . trim(str_replace($basePath, '', $requestUri), '/');
$method        = strtoupper($_SERVER['REQUEST_METHOD'] ?? 'GET');

// Soporte para _method (PUT, DELETE desde formularios HTML)
if ($method === 'POST' && isset($_POST['_method'])) {
    $method = strtoupper($_POST['_method']);
}

// ── Tabla de rutas ────────────────────────────────────────────
// Formato: [MÉTODO, patrón_regex, Controlador, acción]
$routes = [

    // ── Auth ──────────────────────────────────────────────────
    ['GET',    '#^/$#',                          'AuthController',      'loginForm'],
    ['GET',    '#^/login$#',                     'AuthController',      'loginForm'],
    ['POST',   '#^/login$#',                     'AuthController',      'login'],
    ['GET',    '#^/logout$#',                    'AuthController',      'logout'],
    ['GET',    '#^/forgot-password$#',           'AuthController',      'forgotPasswordForm'],
    ['POST',   '#^/forgot-password$#',           'AuthController',      'forgotPassword'],
    ['GET',    '#^/reset-password$#',            'AuthController',      'resetPasswordForm'],
    ['POST',   '#^/reset-password$#',            'AuthController',      'resetPassword'],

    // ── Dashboard ─────────────────────────────────────────────
    ['GET',    '#^/dashboard$#',                 'DashboardController', 'index'],

    // ── Planes (SuperAdmin) ───────────────────────────────────
    ['GET',    '#^/planes$#',                    'PlanController',      'index'],
    ['GET',    '#^/planes/create$#',             'PlanController',      'create'],
    ['POST',   '#^/planes$#',                    'PlanController',      'store'],
    ['GET',    '#^/planes/(\d+)/edit$#',         'PlanController',      'edit'],
    ['POST',   '#^/planes/(\d+)$#',              'PlanController',      'update'],
    ['POST',   '#^/planes/(\d+)/delete$#',       'PlanController',      'destroy'],

    // ── Empresas (SuperAdmin) ─────────────────────────────────
    ['GET',    '#^/empresas$#',                  'EmpresaController',   'index'],
    ['GET',    '#^/empresas/create$#',           'EmpresaController',   'create'],
    ['POST',   '#^/empresas$#',                  'EmpresaController',   'store'],
    ['GET',    '#^/empresas/(\d+)$#',            'EmpresaController',   'show'],
    ['GET',    '#^/empresas/(\d+)/edit$#',       'EmpresaController',   'edit'],
    ['POST',   '#^/empresas/(\d+)$#',            'EmpresaController',   'update'],
    ['POST',   '#^/empresas/(\d+)/delete$#',     'EmpresaController',   'destroy'],

    // ── Licencias ─────────────────────────────────────────────
    ['GET',    '#^/licencias$#',                 'LicenciaController',  'index'],
    ['GET',    '#^/licencias/create$#',          'LicenciaController',  'create'],
    ['POST',   '#^/licencias$#',                 'LicenciaController',  'store'],
    ['POST',   '#^/licencias/(\d+)/suspender$#', 'LicenciaController',  'suspender'],
    ['POST',   '#^/licencias/(\d+)/cancelar$#',  'LicenciaController',  'cancelar'],

    // ── Sucursales ────────────────────────────────────────────
    ['GET',    '#^/sucursales$#',                'SucursalController',  'index'],
    ['GET',    '#^/sucursales/create$#',         'SucursalController',  'create'],
    ['POST',   '#^/sucursales$#',                'SucursalController',  'store'],
    ['GET',    '#^/sucursales/(\d+)/edit$#',     'SucursalController',  'edit'],
    ['POST',   '#^/sucursales/(\d+)$#',          'SucursalController',  'update'],
    ['POST',   '#^/sucursales/(\d+)/delete$#',   'SucursalController',  'destroy'],

    // ── Roles ─────────────────────────────────────────────────
    ['GET',    '#^/roles$#',                     'RolController',       'index'],
    ['GET',    '#^/roles/create$#',              'RolController',       'create'],
    ['POST',   '#^/roles$#',                     'RolController',       'store'],
    ['GET',    '#^/roles/(\d+)/edit$#',          'RolController',       'edit'],
    ['POST',   '#^/roles/(\d+)$#',               'RolController',       'update'],
    ['POST',   '#^/roles/(\d+)/delete$#',        'RolController',       'destroy'],

    // ── Usuarios ──────────────────────────────────────────────
    ['GET',    '#^/usuarios$#',                  'UsuarioController',   'index'],
    ['GET',    '#^/usuarios/create$#',           'UsuarioController',   'create'],
    ['POST',   '#^/usuarios$#',                  'UsuarioController',   'store'],
    ['GET',    '#^/usuarios/(\d+)/edit$#',       'UsuarioController',   'edit'],
    ['POST',   '#^/usuarios/(\d+)$#',            'UsuarioController',   'update'],
    ['POST',   '#^/usuarios/(\d+)/delete$#',     'UsuarioController',   'destroy'],

    // ── Pacientes ─────────────────────────────────────────────
    ['GET',    '#^/pacientes$#',                 'PacienteController',  'index'],
    ['GET',    '#^/pacientes/create$#',          'PacienteController',  'create'],
    ['POST',   '#^/pacientes$#',                 'PacienteController',  'store'],
    ['GET',    '#^/pacientes/(\d+)$#',           'PacienteController',  'show'],
    ['GET',    '#^/pacientes/(\d+)/edit$#',      'PacienteController',  'edit'],
    ['POST',   '#^/pacientes/(\d+)$#',           'PacienteController',  'update'],
    ['POST',   '#^/pacientes/(\d+)/delete$#',    'PacienteController',  'destroy'],
    ['GET',    '#^/pacientes/buscar$#',          'PacienteController',  'buscar'],

    // ── Recetas ───────────────────────────────────────────────
    ['GET',    '#^/recetas$#',                   'RecetaController',    'index'],
    ['GET',    '#^/recetas/create$#',            'RecetaController',    'create'],
    ['POST',   '#^/recetas$#',                   'RecetaController',    'store'],
    ['GET',    '#^/recetas/(\d+)$#',             'RecetaController',    'show'],
    ['GET',    '#^/recetas/(\d+)/print$#',       'RecetaController',    'print'],
    ['POST',   '#^/recetas/(\d+)/delete$#',      'RecetaController',    'destroy'],

    // ── Productos ─────────────────────────────────────────────
    ['GET',    '#^/productos$#',                 'ProductoController',  'index'],
    ['GET',    '#^/productos/create$#',          'ProductoController',  'create'],
    ['POST',   '#^/productos$#',                 'ProductoController',  'store'],
    ['GET',    '#^/productos/(\d+)/edit$#',      'ProductoController',  'edit'],
    ['POST',   '#^/productos/(\d+)$#',           'ProductoController',  'update'],
    ['POST',   '#^/productos/(\d+)/delete$#',    'ProductoController',  'destroy'],
    ['GET',    '#^/productos/buscar$#',          'ProductoController',  'buscar'],

    // ── Stock ─────────────────────────────────────────────────
    ['GET',    '#^/stock$#',                     'StockController',     'index'],
    ['GET',    '#^/stock/movimientos$#',         'StockController',     'movimientos'],
    ['GET',    '#^/stock/ajuste$#',              'StockController',     'movimientos'],
    ['POST',   '#^/stock/ajuste$#',              'StockController',     'ajuste'],
    ['GET',    '#^/stock/kardex$#',              'StockController',     'kardex'],

    // ── Ventas ────────────────────────────────────────────────
    ['GET',    '#^/ventas$#',                    'VentaController',     'index'],
    ['GET',    '#^/ventas/create$#',             'VentaController',     'create'],
    ['POST',   '#^/ventas$#',                    'VentaController',     'store'],
    ['GET',    '#^/ventas/(\d+)$#',              'VentaController',     'show'],
    ['GET',    '#^/ventas/(\d+)/print$#',        'VentaController',     'print'],
    ['POST',   '#^/ventas/(\d+)/anular$#',       'VentaController',     'anular'],

    // ── Caja ──────────────────────────────────────────────────
    ['GET',    '#^/caja$#',                      'CajaController',      'index'],
    ['GET',    '#^/caja/apertura$#',             'CajaController',      'apertura'],
    ['POST',   '#^/caja/apertura$#',             'CajaController',      'abrir'],
    ['GET',    '#^/caja/cierre$#',               'CajaController',      'cierre'],
    ['POST',   '#^/caja/cierre$#',               'CajaController',      'cerrar'],
    ['GET',    '#^/caja/(\d+)$#',                'CajaController',      'show'],
    ['POST',   '#^/caja/movimiento$#',           'CajaController',      'movimiento'],

    // ── Reportes ──────────────────────────────────────────────
    ['GET',    '#^/reportes$#',                  'ReporteController',   'index'],
    ['GET',    '#^/reportes/ventas$#',           'ReporteController',   'ventas'],
    ['GET',    '#^/reportes/ventas/export$#',    'ReporteController',   'exportVentas'],
    ['GET',    '#^/reportes/stock$#',            'ReporteController',   'stock'],
    ['GET',    '#^/reportes/stock/export$#',     'ReporteController',   'exportStock'],
    ['GET',    '#^/reportes/caja$#',             'ReporteController',   'caja'],
    ['GET',    '#^/reportes/caja/export$#',      'ReporteController',   'exportCaja'],

    // ── Categorías ────────────────────────────────────────────
    ['GET',    '#^/categorias$#',                'CategoriaController', 'index'],
    ['GET',    '#^/categorias/create$#',         'CategoriaController', 'create'],
    ['POST',   '#^/categorias$#',                'CategoriaController', 'store'],
    ['GET',    '#^/categorias/(\d+)/edit$#',     'CategoriaController', 'edit'],
    ['POST',   '#^/categorias/(\d+)$#',          'CategoriaController', 'update'],
    ['POST',   '#^/categorias/(\d+)/delete$#',   'CategoriaController', 'destroy'],

    // ── Marcas ────────────────────────────────────────────────
    ['GET',    '#^/marcas$#',                    'MarcaController',     'index'],
    ['GET',    '#^/marcas/create$#',             'MarcaController',     'create'],
    ['POST',   '#^/marcas$#',                    'MarcaController',     'store'],
    ['GET',    '#^/marcas/(\d+)/edit$#',         'MarcaController',     'edit'],
    ['POST',   '#^/marcas/(\d+)$#',              'MarcaController',     'update'],
    ['POST',   '#^/marcas/(\d+)/delete$#',       'MarcaController',     'destroy'],

    // ── Proveedores ───────────────────────────────────────────
    ['GET',    '#^/proveedores$#',               'ProveedorController', 'index'],
    ['GET',    '#^/proveedores/create$#',        'ProveedorController', 'create'],
    ['POST',   '#^/proveedores$#',               'ProveedorController', 'store'],
    ['GET',    '#^/proveedores/(\d+)/edit$#',    'ProveedorController', 'edit'],
    ['POST',   '#^/proveedores/(\d+)$#',         'ProveedorController', 'update'],
    ['POST',   '#^/proveedores/(\d+)/delete$#',  'ProveedorController', 'destroy'],

    // ── Formas de Pago ────────────────────────────────────────
    ['GET',    '#^/formas-pago$#',               'FormaPagoController', 'index'],
    ['GET',    '#^/formas-pago/create$#',        'FormaPagoController', 'create'],
    ['POST',   '#^/formas-pago$#',               'FormaPagoController', 'store'],
    ['GET',    '#^/formas-pago/(\d+)/edit$#',    'FormaPagoController', 'edit'],
    ['POST',   '#^/formas-pago/(\d+)$#',         'FormaPagoController', 'update'],
    ['POST',   '#^/formas-pago/(\d+)/delete$#',  'FormaPagoController', 'destroy'],

    // ── Laboratorio ───────────────────────────────────────────
    ['GET',    '#^/laboratorio$#',               'LaboratorioController', 'index'],
    ['GET',    '#^/laboratorio/create$#',        'LaboratorioController', 'create'],
    ['POST',   '#^/laboratorio$#',               'LaboratorioController', 'store'],
    ['GET',    '#^/laboratorio/(\d+)$#',         'LaboratorioController', 'show'],
    ['GET',    '#^/laboratorio/(\d+)/edit$#',    'LaboratorioController', 'edit'],
    ['POST',   '#^/laboratorio/(\d+)$#',         'LaboratorioController', 'update'],
    ['POST',   '#^/laboratorio/(\d+)/estado$#',  'LaboratorioController', 'cambiarEstado'],

    // ── Agenda ────────────────────────────────────────────────
    ['GET',    '#^/agenda$#',                    'AgendaController',    'index'],
    ['GET',    '#^/agenda/create$#',             'AgendaController',    'create'],
    ['POST',   '#^/agenda$#',                    'AgendaController',    'store'],
    ['GET',    '#^/agenda/(\d+)/edit$#',         'AgendaController',    'edit'],
    ['POST',   '#^/agenda/(\d+)$#',              'AgendaController',    'update'],
    ['POST',   '#^/agenda/(\d+)/estado$#',       'AgendaController',    'cambiarEstado'],
    ['POST',   '#^/agenda/(\d+)/delete$#',       'AgendaController',    'destroy'],

    // ── Perfil ────────────────────────────────────────────────
    ['GET',    '#^/perfil$#',                    'PerfilController',    'index'],
    ['POST',   '#^/perfil$#',                    'PerfilController',    'update'],

    // ── Audit Logs ────────────────────────────────────────────
    ['GET',    '#^/audit-logs$#',                'AuditLogController',  'index'],
];

// ── Resolver ruta ─────────────────────────────────────────────
$matched    = false;
$params     = [];

foreach ($routes as [$routeMethod, $pattern, $controller, $action]) {
    if ($routeMethod !== $method) continue;
    if (preg_match($pattern, $route, $matches)) {
        $params  = array_map('intval', array_slice($matches, 1));
        $matched = true;

        // Middleware de autenticación (excepto login y recuperación de contraseña)
        if (!in_array($action, ['loginForm', 'login', 'forgotPasswordForm', 'forgotPassword', 'resetPasswordForm', 'resetPassword'])) {
            require_once APP_PATH . '/middleware/AuthMiddleware.php';
            AuthMiddleware::handle();
        }

        // Instanciar controlador y llamar acción
        if (!class_exists($controller)) {
            $controllerFile = APP_PATH . '/controllers/' . $controller . '.php';
            if (!file_exists($controllerFile)) {
                http_response_code(500);
                die("Controlador no encontrado: $controller");
            }
            require_once $controllerFile;
        }

        $ctrl = new $controller();
        if (!method_exists($ctrl, $action)) {
            http_response_code(500);
            die("Acción no encontrada: $controller::$action");
        }

        call_user_func_array([$ctrl, $action], $params);
        break;
    }
}

// ── 404 ───────────────────────────────────────────────────────
if (!$matched) {
    http_response_code(404);
    if (isAjax()) {
        jsonResponse(['error' => 'Ruta no encontrada'], 404);
    }
    // Vista 404 simple
    echo '<!DOCTYPE html><html lang="es"><head><meta charset="UTF-8">
    <title>404 - Página no encontrada | OptiCore</title>
    <link rel="stylesheet" href="' . BASE_URL . '/assets/css/app.css">
    </head><body class="bg-light d-flex align-items-center justify-content-center" style="min-height:100vh">
    <div class="text-center">
        <h1 class="display-1 fw-bold text-primary">404</h1>
        <p class="fs-4 text-muted">Página no encontrada</p>
        <a href="' . BASE_URL . '/dashboard" class="btn btn-primary">Ir al Dashboard</a>
    </div></body></html>';
}
