<?php
/**
 * OptiCore SaaS - Producto Model
 */

class Producto extends BaseModel
{
    protected string $table = 'productos';

    public function getAll(int $page = 1, string $search = '', string $tipo = '', int $categoriaId = 0): array
    {
        $params = [$this->empresaScope()];
        $where  = 'p.empresa_id = ?';

        if ($search) {
            $where   .= ' AND (p.nombre LIKE ? OR p.sku LIKE ? OR p.codigo_barras LIKE ?)';
            $s        = "%$search%";
            $params[] = $s; $params[] = $s; $params[] = $s;
        }
        if ($tipo) {
            $where   .= ' AND p.tipo = ?';
            $params[] = $tipo;
        }
        if ($categoriaId) {
            $where   .= ' AND p.categoria_id = ?';
            $params[] = $categoriaId;
        }

        $sql = "SELECT p.*,
                       c.nombre as categoria_nombre,
                       m.nombre as marca_nombre,
                       pr.nombre as proveedor_nombre
                FROM productos p
                LEFT JOIN categorias c ON p.categoria_id = c.id
                LEFT JOIN marcas m ON p.marca_id = m.id
                LEFT JOIN proveedores pr ON p.proveedor_id = pr.id
                WHERE $where
                ORDER BY p.nombre ASC";

        return $this->db->paginate($sql, $params, $page);
    }

    public function getById(int $id): array|false
    {
        return $this->db->fetchOne(
            "SELECT p.*,
                    c.nombre as categoria_nombre,
                    m.nombre as marca_nombre,
                    pr.nombre as proveedor_nombre
             FROM productos p
             LEFT JOIN categorias c ON p.categoria_id = c.id
             LEFT JOIN marcas m ON p.marca_id = m.id
             LEFT JOIN proveedores pr ON p.proveedor_id = pr.id
             WHERE p.id = ? AND p.empresa_id = ?",
            [$id, $this->empresaScope()]
        );
    }

    public function getActivos(): array
    {
        return $this->db->fetchAll(
            "SELECT p.*, c.nombre as categoria_nombre, m.nombre as marca_nombre
             FROM productos p
             LEFT JOIN categorias c ON p.categoria_id = c.id
             LEFT JOIN marcas m ON p.marca_id = m.id
             WHERE p.empresa_id = ? AND p.estado = 'activo'
             ORDER BY p.nombre",
            [$this->empresaScope()]
        );
    }

    public function search(string $q, int $sucursalId = 0): array
    {
        $s      = "%$q%";
        $params = [$this->empresaScope(), $s, $s, $s];

        $stockJoin = '';
        $stockSel  = '0 as stock_actual';
        if ($sucursalId) {
            $stockJoin = "LEFT JOIN stock st ON st.producto_id = p.id AND st.sucursal_id = $sucursalId";
            $stockSel  = 'COALESCE(st.cantidad, 0) as stock_actual';
        }

        return $this->db->fetchAll(
            "SELECT p.id, p.nombre, p.sku, p.precio_venta, p.tipo, $stockSel
             FROM productos p $stockJoin
             WHERE p.empresa_id = ? AND p.estado = 'activo'
             AND (p.nombre LIKE ? OR p.sku LIKE ? OR p.codigo_barras LIKE ?)
             ORDER BY p.nombre LIMIT 20",
            $params
        );
    }

    public function getStockCritico(?int $sucursalId = null): array
    {
        $params = [$this->empresaScope()];
        $sucWhere = '';
        if ($sucursalId) {
            $sucWhere  = 'AND st.sucursal_id = ?';
            $params[]  = $sucursalId;
        }

        return $this->db->fetchAll(
            "SELECT p.*, st.cantidad as stock_actual, st.sucursal_id,
                    s.nombre as sucursal_nombre
             FROM productos p
             JOIN stock st ON st.producto_id = p.id
             JOIN sucursales s ON st.sucursal_id = s.id
             WHERE p.empresa_id = ? AND p.estado = 'activo'
             AND st.cantidad <= p.stock_minimo $sucWhere
             ORDER BY st.cantidad ASC",
            $params
        );
    }

    public function getMasVendidos(string $desde, string $hasta, int $limit = 10): array
    {
        return $this->db->fetchAll(
            "SELECT p.nombre, p.tipo, SUM(vi.cantidad) as total_vendido,
                    SUM(vi.subtotal) as total_ingresos
             FROM venta_items vi
             JOIN ventas v ON vi.venta_id = v.id
             JOIN productos p ON vi.producto_id = p.id
             WHERE v.empresa_id = ? AND v.estado = 'confirmada'
             AND DATE(v.fecha) BETWEEN ? AND ?
             GROUP BY vi.producto_id
             ORDER BY total_vendido DESC
             LIMIT ?",
            [$this->empresaScope(), $desde, $hasta, $limit]
        );
    }

    public function getCategorias(): array
    {
        return $this->db->fetchAll(
            "SELECT * FROM categorias WHERE empresa_id = ? AND estado = 'activa' ORDER BY nombre",
            [$this->empresaScope()]
        );
    }

    public function getMarcas(): array
    {
        return $this->db->fetchAll(
            "SELECT * FROM marcas WHERE empresa_id = ? AND estado = 'activa' ORDER BY nombre",
            [$this->empresaScope()]
        );
    }

    public function getProveedores(): array
    {
        return $this->db->fetchAll(
            "SELECT * FROM proveedores WHERE empresa_id = ? AND estado = 'activo' ORDER BY nombre",
            [$this->empresaScope()]
        );
    }

    // ── Listar con paginación (usado por el controlador) ──────
    public function listar(int $page = 1, int $perPage = ITEMS_PER_PAGE, ?string $search = null, ?string $tipo = null): array
    {
        $result = $this->getAll($page, $search ?? '', $tipo ?? '');
        return [
            'data'       => $result['data'],
            'pagination' => $result,
        ];
    }

    // ── Buscar para venta (AJAX) ──────────────────────────────
    public function buscarParaVenta(string $q, int $sucursalId = 0): array
    {
        return $this->search($q, $sucursalId);
    }

    // ── Crear producto ────────────────────────────────────────
    public function crear(array $data): int
    {
        return $this->db->insert('productos', [
            'empresa_id'    => $this->empresaScope(),
            'categoria_id'  => !empty($data['categoria_id']) ? $data['categoria_id'] : null,
            'marca_id'      => !empty($data['marca_id'])     ? $data['marca_id']     : null,
            'proveedor_id'  => !empty($data['proveedor_id']) ? $data['proveedor_id'] : null,
            'nombre'        => trim($data['nombre']),
            'descripcion'   => trim($data['descripcion'] ?? '') ?: null,
            'sku'           => trim($data['sku'] ?? '') ?: null,
            'codigo_barras' => trim($data['codigo_barras'] ?? '') ?: null,
            'tipo'          => $data['tipo'] ?? 'producto',
            'precio_costo'  => $data['precio_costo'] ?? 0,
            'precio_venta'  => $data['precio_venta'],
            'stock_minimo'  => $data['stock_minimo'] ?? 0,
            'imagen'        => $data['imagen'] ?? null,
            'estado'        => $data['estado'] ?? 'activo',
        ]);
    }

    // ── Actualizar producto ───────────────────────────────────
    public function actualizar(int $id, array $data): int
    {
        $campos = [
            'categoria_id'  => !empty($data['categoria_id']) ? $data['categoria_id'] : null,
            'marca_id'      => !empty($data['marca_id'])     ? $data['marca_id']     : null,
            'proveedor_id'  => !empty($data['proveedor_id']) ? $data['proveedor_id'] : null,
            'nombre'        => trim($data['nombre']),
            'descripcion'   => trim($data['descripcion'] ?? '') ?: null,
            'sku'           => trim($data['sku'] ?? '') ?: null,
            'codigo_barras' => trim($data['codigo_barras'] ?? '') ?: null,
            'tipo'          => $data['tipo'] ?? 'producto',
            'precio_costo'  => $data['precio_costo'] ?? 0,
            'precio_venta'  => $data['precio_venta'],
            'stock_minimo'  => $data['stock_minimo'] ?? 0,
            'estado'        => $data['estado'] ?? 'activo',
        ];
        if (array_key_exists('imagen', $data)) {
            $campos['imagen'] = $data['imagen'];
        }
        return $this->update($id, $campos);
    }
}
