<?php
/**
 * OptiCore SaaS - Database PDO Wrapper (Singleton)
 */

class Database
{
    private static ?Database $instance = null;
    private PDO $pdo;

    private function __construct()
    {
        $config = require CONFIG_PATH . '/database.php';

        $dsn = sprintf(
            'mysql:host=%s;port=%d;dbname=%s;charset=%s',
            $config['host'],
            $config['port'],
            $config['dbname'],
            $config['charset']
        );

        try {
            $this->pdo = new PDO($dsn, $config['username'], $config['password'], $config['options']);
        } catch (PDOException $e) {
            if (APP_ENV === 'development') {
                die('<div style="font-family:monospace;background:#fee;padding:20px;border:1px solid red;">
                    <strong>Error de conexión a la base de datos:</strong><br>' . $e->getMessage() . '
                </div>');
            }
            die('Error de conexión. Contacte al administrador.');
        }
    }

    public static function getInstance(): self
    {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    // ── Prevenir clonación ────────────────────────────────────
    private function __clone() {}

    // ── Query base ────────────────────────────────────────────
    public function query(string $sql, array $params = []): PDOStatement
    {
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute($params);
        return $stmt;
    }

    // ── Obtener todos los registros ───────────────────────────
    public function fetchAll(string $sql, array $params = []): array
    {
        return $this->query($sql, $params)->fetchAll();
    }

    // ── Obtener un registro ───────────────────────────────────
    public function fetchOne(string $sql, array $params = []): array|false
    {
        return $this->query($sql, $params)->fetch();
    }

    // ── Obtener valor escalar ─────────────────────────────────
    public function fetchColumn(string $sql, array $params = []): mixed
    {
        return $this->query($sql, $params)->fetchColumn();
    }

    // ── Insertar registro ─────────────────────────────────────
    public function insert(string $table, array $data): int
    {
        $columns = implode(', ', array_map(fn($k) => "`$k`", array_keys($data)));
        $placeholders = implode(', ', array_fill(0, count($data), '?'));
        $sql = "INSERT INTO `$table` ($columns) VALUES ($placeholders)";
        $this->query($sql, array_values($data));
        return (int) $this->pdo->lastInsertId();
    }

    // ── Actualizar registro ───────────────────────────────────
    public function update(string $table, array $data, array $where): int
    {
        $set   = implode(', ', array_map(fn($k) => "`$k` = ?", array_keys($data)));
        $cond  = implode(' AND ', array_map(fn($k) => "`$k` = ?", array_keys($where)));
        $sql   = "UPDATE `$table` SET $set WHERE $cond";
        $stmt  = $this->query($sql, [...array_values($data), ...array_values($where)]);
        return $stmt->rowCount();
    }

    // ── Eliminar registro ─────────────────────────────────────
    public function delete(string $table, array $where): int
    {
        $cond = implode(' AND ', array_map(fn($k) => "`$k` = ?", array_keys($where)));
        $sql  = "DELETE FROM `$table` WHERE $cond";
        return $this->query($sql, array_values($where))->rowCount();
    }

    // ── Verificar existencia ──────────────────────────────────
    public function exists(string $table, array $where): bool
    {
        $cond = implode(' AND ', array_map(fn($k) => "`$k` = ?", array_keys($where)));
        $sql  = "SELECT 1 FROM `$table` WHERE $cond LIMIT 1";
        return (bool) $this->fetchColumn($sql, array_values($where));
    }

    // ── Contar registros ──────────────────────────────────────
    public function count(string $table, array $where = []): int
    {
        if (empty($where)) {
            return (int) $this->fetchColumn("SELECT COUNT(*) FROM `$table`");
        }
        $cond = implode(' AND ', array_map(fn($k) => "`$k` = ?", array_keys($where)));
        return (int) $this->fetchColumn("SELECT COUNT(*) FROM `$table` WHERE $cond", array_values($where));
    }

    // ── Transacciones ─────────────────────────────────────────
    public function beginTransaction(): void
    {
        $this->pdo->beginTransaction();
    }

    public function commit(): void
    {
        $this->pdo->commit();
    }

    public function rollBack(): void
    {
        $this->pdo->rollBack();
    }

    public function inTransaction(): bool
    {
        return $this->pdo->inTransaction();
    }

    // ── Último ID insertado ───────────────────────────────────
    public function lastInsertId(): int
    {
        return (int) $this->pdo->lastInsertId();
    }

    // ── Paginación ────────────────────────────────────────────
    public function paginate(string $sql, array $params, int $page, int $perPage = ITEMS_PER_PAGE): array
    {
        // Total
        $countSql = "SELECT COUNT(*) FROM ($sql) AS _count_query";
        $total    = (int) $this->fetchColumn($countSql, $params);

        // Datos paginados
        $offset = ($page - 1) * $perPage;
        $data   = $this->fetchAll("$sql LIMIT $perPage OFFSET $offset", $params);

        return [
            'data'         => $data,
            'total'        => $total,
            'per_page'     => $perPage,
            'current_page' => $page,
            'last_page'    => (int) ceil($total / $perPage),
            'from'         => $offset + 1,
            'to'           => min($offset + $perPage, $total),
        ];
    }

    // ── PDO directo (para casos especiales) ───────────────────
    public function getPdo(): PDO
    {
        return $this->pdo;
    }
}
