<?php
/**
 * OptiCore SaaS - Cita Model (Agenda)
 */

class Cita extends BaseModel
{
    protected string $table = 'citas';

    public function getAll(int $page = 1, array $filtros = []): array
    {
        $params = [$this->empresaScope()];
        $where  = 'c.empresa_id = ?';

        if (!empty($filtros['fecha'])) {
            $where   .= ' AND c.fecha = ?';
            $params[] = $filtros['fecha'];
        }
        if (!empty($filtros['estado'])) {
            $where   .= ' AND c.estado = ?';
            $params[] = $filtros['estado'];
        }
        if (!empty($filtros['desde'])) {
            $where   .= ' AND c.fecha >= ?';
            $params[] = $filtros['desde'];
        }
        if (!empty($filtros['hasta'])) {
            $where   .= ' AND c.fecha <= ?';
            $params[] = $filtros['hasta'];
        }
        if (!empty($filtros['q'])) {
            $where   .= ' AND (c.nombre_paciente LIKE ? OR c.telefono LIKE ?)';
            $s        = '%' . $filtros['q'] . '%';
            $params[] = $s; $params[] = $s;
        }

        $sql = "SELECT c.*,
                       CONCAT(p.nombre, ' ', p.apellido) AS paciente_nombre_real,
                       p.telefono AS paciente_telefono_real,
                       s.nombre AS sucursal_nombre
                FROM `{$this->table}` c
                LEFT JOIN pacientes p  ON c.paciente_id  = p.id
                LEFT JOIN sucursales s ON c.sucursal_id  = s.id
                WHERE $where
                ORDER BY c.fecha ASC, c.hora ASC";

        return $this->db->paginate($sql, $params, $page);
    }

    public function getById(int $id): array|false
    {
        return $this->db->fetchOne(
            "SELECT c.*,
                    CONCAT(p.nombre, ' ', p.apellido) AS paciente_nombre_real,
                    p.telefono AS paciente_telefono_real,
                    s.nombre AS sucursal_nombre
             FROM `{$this->table}` c
             LEFT JOIN pacientes p  ON c.paciente_id  = p.id
             LEFT JOIN sucursales s ON c.sucursal_id  = s.id
             WHERE c.id = ? AND c.empresa_id = ?",
            [$id, $this->empresaScope()]
        );
    }

    public function getDelDia(string $fecha = ''): array
    {
        $fecha = $fecha ?: date('Y-m-d');
        return $this->db->fetchAll(
            "SELECT c.*,
                    CONCAT(p.nombre, ' ', p.apellido) AS paciente_nombre_real
             FROM `{$this->table}` c
             LEFT JOIN pacientes p ON c.paciente_id = p.id
             WHERE c.empresa_id = ? AND c.fecha = ?
             ORDER BY c.hora ASC",
            [$this->empresaScope(), $fecha]
        );
    }

    public function getProximas(int $dias = 7): array
    {
        return $this->db->fetchAll(
            "SELECT c.*,
                    CONCAT(p.nombre, ' ', p.apellido) AS paciente_nombre_real
             FROM `{$this->table}` c
             LEFT JOIN pacientes p ON c.paciente_id = p.id
             WHERE c.empresa_id = ?
               AND c.fecha BETWEEN CURDATE() AND DATE_ADD(CURDATE(), INTERVAL ? DAY)
               AND c.estado IN ('pendiente','confirmada')
             ORDER BY c.fecha ASC, c.hora ASC",
            [$this->empresaScope(), $dias]
        );
    }

    public function crear(array $data): int
    {
        $sucursalId = Auth::sucursalId() ?? $data['sucursal_id'] ?? null;

        return $this->db->insert($this->table, [
            'empresa_id'      => $this->empresaScope(),
            'sucursal_id'     => $sucursalId,
            'paciente_id'     => !empty($data['paciente_id']) ? (int) $data['paciente_id'] : null,
            'usuario_id'      => Auth::id(),
            'fecha'           => $data['fecha'],
            'hora'            => $data['hora'],
            'duracion'        => (int) ($data['duracion'] ?? 30),
            'tipo'            => $data['tipo'] ?? 'optometria',
            'estado'          => $data['estado'] ?? 'pendiente',
            'nombre_paciente' => trim($data['nombre_paciente'] ?? '') ?: null,
            'telefono'        => trim($data['telefono'] ?? '') ?: null,
            'observaciones'   => trim($data['observaciones'] ?? '') ?: null,
        ]);
    }

    public function actualizar(int $id, array $data): int
    {
        return $this->update($id, [
            'paciente_id'     => !empty($data['paciente_id']) ? (int) $data['paciente_id'] : null,
            'fecha'           => $data['fecha'],
            'hora'            => $data['hora'],
            'duracion'        => (int) ($data['duracion'] ?? 30),
            'tipo'            => $data['tipo'] ?? 'optometria',
            'estado'          => $data['estado'] ?? 'pendiente',
            'nombre_paciente' => trim($data['nombre_paciente'] ?? '') ?: null,
            'telefono'        => trim($data['telefono'] ?? '') ?: null,
            'observaciones'   => trim($data['observaciones'] ?? '') ?: null,
        ]);
    }

    public function cambiarEstado(int $id, string $estado): int
    {
        return $this->update($id, ['estado' => $estado]);
    }

    public function contarPorEstado(string $fecha = ''): array
    {
        $fecha  = $fecha ?: date('Y-m-d');
        $rows   = $this->db->fetchAll(
            "SELECT estado, COUNT(*) as total FROM `{$this->table}`
             WHERE empresa_id = ? AND fecha = ?
             GROUP BY estado",
            [$this->empresaScope(), $fecha]
        );
        $result = [];
        foreach ($rows as $r) {
            $result[$r['estado']] = (int) $r['total'];
        }
        return $result;
    }
}
