<?php
/**
 * OptiCore SaaS - Funciones Helper Globales
 */

// ── Redirección ───────────────────────────────────────────────
function redirect(string $path): never
{
    $url = str_starts_with($path, 'http') ? $path : BASE_URL . $path;
    header("Location: $url");
    exit;
}

// ── Flash messages ────────────────────────────────────────────
function flash(string $type, string $message): void
{
    $_SESSION['flash'][] = ['type' => $type, 'message' => $message];
}

function getFlash(): array
{
    $flash = $_SESSION['flash'] ?? [];
    unset($_SESSION['flash']);
    return $flash;
}

// ── Respuesta JSON ────────────────────────────────────────────
function jsonResponse(mixed $data, int $code = 200): never
{
    http_response_code($code);
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode($data, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    exit;
}

// ── Detectar AJAX ─────────────────────────────────────────────
function isAjax(): bool
{
    return isset($_SERVER['HTTP_X_REQUESTED_WITH'])
        && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) === 'xmlhttprequest';
}

// ── Sanitización ──────────────────────────────────────────────
function clean(mixed $value): string
{
    return htmlspecialchars(trim((string) $value), ENT_QUOTES, 'UTF-8');
}

function cleanArray(array $data): array
{
    return array_map('clean', $data);
}

// ── Obtener POST/GET con default ──────────────────────────────
function post(string $key, mixed $default = ''): mixed
{
    return $_POST[$key] ?? $default;
}

function get(string $key, mixed $default = ''): mixed
{
    return $_GET[$key] ?? $default;
}

function postInt(string $key, int $default = 0): int
{
    return (int) ($_POST[$key] ?? $default);
}

function getInt(string $key, int $default = 0): int
{
    return (int) ($_GET[$key] ?? $default);
}

// ── Renderizar vista ──────────────────────────────────────────
function view(string $view, array $data = [], string $layout = 'main'): void
{
    extract($data);
    $viewFile = VIEWS_PATH . '/' . str_replace('.', '/', $view) . '.php';

    if (!file_exists($viewFile)) {
        die("Vista no encontrada: $viewFile");
    }

    // Capturar contenido de la vista
    ob_start();
    require $viewFile;
    $content = ob_get_clean();

    // Renderizar layout
    $layoutFile = VIEWS_PATH . '/layouts/' . $layout . '.php';
    if (file_exists($layoutFile)) {
        require $layoutFile;
    } else {
        echo $content;
    }
}

// ── Formateo de moneda ────────────────────────────────────────
function formatMoney(float $amount, string $currency = 'CLP'): string
{
    if ($currency === 'CLP') {
        return '$' . number_format($amount, 0, ',', '.');
    }
    return number_format($amount, 2, '.', ',');
}

// ── Formateo de fecha ─────────────────────────────────────────
function formatDate(string $date, string $format = 'd/m/Y'): string
{
    if (empty($date) || $date === '0000-00-00') return '-';
    return date($format, strtotime($date));
}

function formatDateTime(string $datetime, string $format = 'd/m/Y H:i'): string
{
    if (empty($datetime)) return '-';
    return date($format, strtotime($datetime));
}

// ── Edad desde fecha de nacimiento ────────────────────────────
function calcularEdad(string $fechaNacimiento): int
{
    return (int) date_diff(date_create($fechaNacimiento), date_create('today'))->y;
}

// ── Paginación HTML ───────────────────────────────────────────
function paginationLinks(array $pagination, string $baseUrl): string
{
    if ($pagination['last_page'] <= 1) return '';

    $html  = '<nav><ul class="pagination pagination-sm mb-0">';
    $page  = $pagination['current_page'];
    $last  = $pagination['last_page'];

    // Anterior
    $html .= '<li class="page-item ' . ($page <= 1 ? 'disabled' : '') . '">';
    $html .= '<a class="page-link" href="' . $baseUrl . '?page=' . ($page - 1) . '">&laquo;</a></li>';

    // Páginas
    $start = max(1, $page - 2);
    $end   = min($last, $page + 2);

    if ($start > 1) {
        $html .= '<li class="page-item"><a class="page-link" href="' . $baseUrl . '?page=1">1</a></li>';
        if ($start > 2) $html .= '<li class="page-item disabled"><span class="page-link">...</span></li>';
    }

    for ($i = $start; $i <= $end; $i++) {
        $active = $i === $page ? 'active' : '';
        $html  .= '<li class="page-item ' . $active . '">';
        $html  .= '<a class="page-link" href="' . $baseUrl . '?page=' . $i . '">' . $i . '</a></li>';
    }

    if ($end < $last) {
        if ($end < $last - 1) $html .= '<li class="page-item disabled"><span class="page-link">...</span></li>';
        $html .= '<li class="page-item"><a class="page-link" href="' . $baseUrl . '?page=' . $last . '">' . $last . '</a></li>';
    }

    // Siguiente
    $html .= '<li class="page-item ' . ($page >= $last ? 'disabled' : '') . '">';
    $html .= '<a class="page-link" href="' . $baseUrl . '?page=' . ($page + 1) . '">&raquo;</a></li>';
    $html .= '</ul></nav>';

    return $html;
}

// ── Badge de estado ───────────────────────────────────────────
function statusBadge(string $estado): string
{
    $map = [
        'activo'      => 'success', 'activa'     => 'success',
        'inactivo'    => 'secondary','inactiva'  => 'secondary',
        'confirmada'  => 'success', 'confirmado' => 'success',
        'pendiente'   => 'warning', 'abierta'    => 'success',
        'cerrada'     => 'secondary','anulada'   => 'danger',
        'anulado'     => 'danger',  'suspendida' => 'warning',
        'expirada'    => 'danger',  'prueba'     => 'info',
        'cotizacion'  => 'info',    'bloqueado'  => 'danger',
        'recibido'    => 'info',    'en_proceso' => 'warning',
        'listo'       => 'primary', 'entregado'  => 'success',
        'cancelado'   => 'danger',  'cancelada'  => 'danger',
        'atendida'    => 'success', 'no_asistio' => 'danger',
        'vencida'     => 'warning',
    ];
    $color = $map[$estado] ?? 'secondary';
    $label = ucfirst(str_replace('_', ' ', $estado));
    return "<span class=\"badge bg-$color\">$label</span>";
}

// ── Generar correlativo ───────────────────────────────────────
function generarCorrelativo(string $prefix, int $numero, int $pad = 4): string
{
    return $prefix . str_pad($numero, $pad, '0', STR_PAD_LEFT);
}

// ── IP del cliente ────────────────────────────────────────────
function clientIp(): string
{
    return $_SERVER['HTTP_X_FORWARDED_FOR']
        ?? $_SERVER['HTTP_CLIENT_IP']
        ?? $_SERVER['REMOTE_ADDR']
        ?? '0.0.0.0';
}

// ── URL actual ────────────────────────────────────────────────
function currentUrl(): string
{
    return BASE_URL . ($_SERVER['REQUEST_URI'] ?? '/');
}

// ── Verificar si URL está activa (para menú) ──────────────────
function isActiveUrl(string $path): string
{
    $current = parse_url($_SERVER['REQUEST_URI'] ?? '/', PHP_URL_PATH);
    return str_contains($current, $path) ? 'active' : '';
}

// ── Subir archivo ─────────────────────────────────────────────
function uploadFile(array $file, string $folder, array $allowedTypes = ['jpg','jpeg','png','pdf']): string|false
{
    if ($file['error'] !== UPLOAD_ERR_OK) return false;

    $ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
    if (!in_array($ext, $allowedTypes)) return false;

    $maxSize = 5 * 1024 * 1024; // 5MB
    if ($file['size'] > $maxSize) return false;

    $dir = UPLOADS_PATH . '/' . $folder;
    if (!is_dir($dir)) mkdir($dir, 0755, true);

    $filename = uniqid() . '_' . time() . '.' . $ext;
    $dest     = $dir . '/' . $filename;

    if (move_uploaded_file($file['tmp_name'], $dest)) {
        return 'uploads/' . $folder . '/' . $filename;
    }
    return false;
}

// ── Truncar texto ─────────────────────────────────────────────
function truncate(string $text, int $length = 50): string
{
    return mb_strlen($text) > $length
        ? mb_substr($text, 0, $length) . '...'
        : $text;
}

// ── Número de receta formateado ───────────────────────────────
function formatReceta(float|null $value): string
{
    if ($value === null) return '-';
    $formatted = number_format(abs($value), 2);
    return $value >= 0 ? "+$formatted" : "-$formatted";
}

// ── Validación básica ─────────────────────────────────────────
function validateRequired(array $data, array $fields): array
{
    $errors = [];
    foreach ($fields as $field => $label) {
        if (empty(trim($data[$field] ?? ''))) {
            $errors[$field] = "$label es requerido.";
        }
    }
    return $errors;
}

// ── Número de página actual ───────────────────────────────────
function currentPage(): int
{
    return max(1, getInt('page', 1));
}

// ── Asset URL ─────────────────────────────────────────────────
function asset(string $path): string
{
    return BASE_URL . '/assets/' . ltrim($path, '/');
}
