<?php
/**
 * OptiCore SaaS - VentaController
 */

class VentaController
{
    private Venta $model;

    public function __construct()
    {
        Auth::requirePermission('ventas.ver');
        $this->model = new Venta();
    }

    // ── GET /ventas ───────────────────────────────────────────
    public function index(): void
    {
        $page      = currentPage();
        $desde     = get('desde');
        $hasta     = get('hasta');
        $estado    = get('estado');
        $resultado = $this->model->listar($page, ITEMS_PER_PAGE, [
            'desde'  => $desde ?: null,
            'hasta'  => $hasta ?: null,
            'estado' => $estado ?: null,
        ]);

        view('ventas.index', [
            'title'      => 'Ventas',
            'ventas'     => $resultado['data'],
            'pagination' => $resultado['pagination'],
            'desde'      => $desde,
            'hasta'      => $hasta,
            'estado'     => $estado,
        ]);
    }

    // ── GET /ventas/create ────────────────────────────────────
    public function create(): void
    {
        Auth::requirePermission('ventas.crear');

        // Verificar caja abierta
        $sucursalId = Auth::sucursalId();
        if ($sucursalId) {
            $caja = $this->getCajaAbierta($sucursalId);
            if (!$caja) {
                flash('error', 'Debes abrir la caja antes de registrar ventas.');
                redirect('/caja/apertura');
            }
        }

        $formasPago = $this->getFormasPago();
        $sucursales = (new Sucursal())->getByEmpresa(Auth::empresaId());

        view('ventas.create', [
            'title'       => 'Nueva Venta',
            'formas_pago' => $formasPago,
            'sucursales'  => $sucursales,
        ]);
    }

    // ── POST /ventas ──────────────────────────────────────────
    public function store(): void
    {
        Auth::requirePermission('ventas.crear');
        csrf_verify();

        // Validar items
        $items = json_decode($_POST['items'] ?? '[]', true);
        if (empty($items)) {
            flash('error', 'Debes agregar al menos un producto.');
            redirect('/ventas/create');
        }

        $sucursalId = Auth::sucursalId();
        $empresaId  = Auth::empresaId();

        if (!$sucursalId || !$empresaId) {
            flash('error', 'No tienes sucursal asignada.');
            redirect('/ventas/create');
        }

        try {
            $ventaData = [
                'empresa_id'    => $empresaId,
                'sucursal_id'   => $sucursalId,
                'usuario_id'    => Auth::id(),
                'paciente_id'   => !empty($_POST['paciente_id']) ? (int) $_POST['paciente_id'] : null,
                'descuento'     => (float) ($_POST['descuento'] ?? 0),
                'observaciones' => $_POST['observaciones'] ?? null,
                'estado'        => $_POST['estado'] ?? 'confirmada',
            ];

            $pagos = json_decode($_POST['pagos'] ?? '[]', true);

            $ventaId = $this->model->crear($ventaData, $items, $pagos);
            AuditLog::log('crear_venta', 'ventas', $ventaId);
            flash('success', 'Venta registrada correctamente.');
            redirect("/ventas/$ventaId");
        } catch (Exception $e) {
            flash('error', 'Error al registrar la venta: ' . $e->getMessage());
            redirect('/ventas/create');
        }
    }

    // ── GET /ventas/{id} ──────────────────────────────────────
    public function show(int $id): void
    {
        $venta = $this->model->getById($id);
        if (!$venta) {
            flash('error', 'Venta no encontrada.');
            redirect('/ventas');
        }

        view('ventas.show', [
            'title' => 'Venta ' . $venta['correlativo'],
            'venta' => $venta,
        ]);
    }

    // ── GET /ventas/{id}/print ────────────────────────────────
    public function print(int $id): void
    {
        Auth::requirePermission('ventas.imprimir');

        $venta = $this->model->getById($id);
        if (!$venta) {
            flash('error', 'Venta no encontrada.');
            redirect('/ventas');
        }

        $empresa = db()->fetchOne(
            "SELECT * FROM empresas WHERE id = ?",
            [$venta['empresa_id']]
        );

        view('ventas.print', [
            'title'   => 'Boleta ' . $venta['correlativo'],
            'venta'   => $venta,
            'empresa' => $empresa,
        ], 'print');
    }

    // ── POST /ventas/{id}/anular ──────────────────────────────
    public function anular(int $id): void
    {
        Auth::requirePermission('ventas.anular');
        csrf_verify();

        $venta = $this->model->getById($id);
        if (!$venta) {
            flash('error', 'Venta no encontrada.');
            redirect('/ventas');
        }

        if ($venta['estado'] === 'anulada') {
            flash('error', 'La venta ya está anulada.');
            redirect("/ventas/$id");
        }

        try {
            $motivo = trim($_POST['motivo'] ?? '');
            $this->model->anular($id, $motivo);
            AuditLog::log('anular_venta', 'ventas', $id, $venta);
            flash('success', 'Venta anulada correctamente.');
        } catch (Exception $e) {
            flash('error', 'Error al anular la venta: ' . $e->getMessage());
        }

        redirect("/ventas/$id");
    }

    // ── Caja abierta de la sucursal ───────────────────────────
    private function getCajaAbierta(int $sucursalId): array|false
    {
        return db()->fetchOne(
            "SELECT * FROM caja_sesiones
             WHERE sucursal_id = ? AND estado = 'abierta'
             ORDER BY fecha_apertura DESC LIMIT 1",
            [$sucursalId]
        );
    }

    // ── Formas de pago activas ────────────────────────────────
    private function getFormasPago(): array
    {
        $empresaId = Auth::empresaId();
        if (!$empresaId) return [];
        return db()->fetchAll(
            "SELECT * FROM formas_pago WHERE empresa_id = ? AND estado = 'activa' ORDER BY nombre",
            [$empresaId]
        );
    }
}
