<?php
/**
 * OptiCore SaaS - AuthController
 */

class AuthController
{
    // ── GET /login ────────────────────────────────────────────
    public function loginForm(): void
    {
        if (Auth::check()) {
            redirect('/dashboard');
        }
        view('auth.login', ['title' => 'Iniciar Sesión'], 'auth');
    }

    // ── POST /login ───────────────────────────────────────────
    public function login(): void
    {
        if (Auth::check()) {
            redirect('/dashboard');
        }

        // Validar CSRF
        csrf_verify();

        $email    = trim(post('email'));
        $password = post('password');

        if (empty($email) || empty($password)) {
            flash('error', 'Email y contraseña son requeridos.');
            redirect('/login');
        }

        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            flash('error', 'Email inválido.');
            redirect('/login');
        }

        if (Auth::attempt($email, $password)) {
            redirect('/dashboard');
        }

        flash('error', 'Credenciales incorrectas o cuenta inactiva.');
        redirect('/login');
    }

    // ── GET /logout ───────────────────────────────────────────
    public function logout(): void
    {
        Auth::logout();
        flash('success', 'Sesión cerrada correctamente.');
        redirect('/login');
    }

    // ── GET /forgot-password ──────────────────────────────────
    public function forgotPasswordForm(): void
    {
        if (Auth::check()) redirect('/dashboard');
        view('auth.forgot_password', ['title' => 'Recuperar Contraseña'], 'auth');
    }

    // ── POST /forgot-password ─────────────────────────────────
    public function forgotPassword(): void
    {
        if (Auth::check()) redirect('/dashboard');
        csrf_verify();

        $email = trim(post('email'));

        if (empty($email) || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
            flash('error', 'Ingresa un email válido.');
            redirect('/forgot-password');
        }

        $db   = db();
        $user = $db->fetchOne(
            "SELECT id, nombre, email FROM usuarios WHERE email = ? AND estado = 'activo' LIMIT 1",
            [$email]
        );

        // Siempre mostrar la misma vista para no revelar si el email existe
        $resetLink = null;

        if ($user) {
            // Generar token seguro (64 hex chars = 32 bytes)
            $token   = bin2hex(random_bytes(32));
            $expira  = date('Y-m-d H:i:s', strtotime('+1 hour'));

            $db->update('usuarios',
                ['token_reset' => $token, 'token_expira' => $expira],
                ['id' => $user['id']]
            );

            $resetLink = BASE_URL . '/reset-password?token=' . $token;
        }

        // En desarrollo: mostrar el enlace en pantalla
        // En producción: enviar por email y mostrar mensaje genérico
        view('auth.forgot_password', [
            'title'     => 'Recuperar Contraseña',
            'resetLink' => $resetLink,
            'email'     => $email,
        ], 'auth');
    }

    // ── GET /reset-password ───────────────────────────────────
    public function resetPasswordForm(): void
    {
        if (Auth::check()) redirect('/dashboard');

        $token = get('token');
        if (empty($token)) {
            flash('error', 'Token de restablecimiento inválido.');
            redirect('/forgot-password');
        }

        $db   = db();
        $user = $db->fetchOne(
            "SELECT id, email FROM usuarios
             WHERE token_reset = ? AND token_expira > NOW() AND estado = 'activo'
             LIMIT 1",
            [$token]
        );

        if (!$user) {
            flash('error', 'El enlace de restablecimiento es inválido o ha expirado.');
            redirect('/forgot-password');
        }

        view('auth.reset_password', [
            'title' => 'Nueva Contraseña',
            'token' => $token,
            'email' => $user['email'],
        ], 'auth');
    }

    // ── POST /reset-password ──────────────────────────────────
    public function resetPassword(): void
    {
        if (Auth::check()) redirect('/dashboard');
        csrf_verify();

        $token    = post('token');
        $password = post('password');
        $confirm  = post('password_confirm');

        if (empty($token) || empty($password) || empty($confirm)) {
            flash('error', 'Todos los campos son requeridos.');
            redirect('/reset-password?token=' . urlencode($token));
        }

        if (strlen($password) < 8) {
            flash('error', 'La contraseña debe tener al menos 8 caracteres.');
            redirect('/reset-password?token=' . urlencode($token));
        }

        if ($password !== $confirm) {
            flash('error', 'Las contraseñas no coinciden.');
            redirect('/reset-password?token=' . urlencode($token));
        }

        $db   = db();
        $user = $db->fetchOne(
            "SELECT id, email FROM usuarios
             WHERE token_reset = ? AND token_expira > NOW() AND estado = 'activo'
             LIMIT 1",
            [$token]
        );

        if (!$user) {
            flash('error', 'El enlace de restablecimiento es inválido o ha expirado.');
            redirect('/forgot-password');
        }

        // Actualizar contraseña y limpiar token
        $db->update('usuarios', [
            'password'     => password_hash($password, PASSWORD_BCRYPT, ['cost' => BCRYPT_COST]),
            'token_reset'  => null,
            'token_expira' => null,
        ], ['id' => $user['id']]);

        AuditLog::log('reset_password', 'usuarios', $user['id']);

        flash('success', 'Contraseña restablecida correctamente. Ya puedes iniciar sesión.');
        redirect('/login');
    }
}
